%% Script that processes the validated questionnaires
clear,clc,close all

%% Read raw scores
T_SSQ = readtable('SSQ.xlsx');
T_SPaRQ = readtable('SPaRQ.xlsx');

%% SSQ subscales: Speech (1-5), Spatial (6-8), Qualities (9-12)
% Noble et al. (2013). doi: 10.3109/14992027.2013.781278

% Get average scores
SSQ = nan(32,4);
SSQ(:,1) = mean(table2array(T_SSQ(:,2:6)),2);       % 1. Speech dimension
SSQ(:,2) = mean(table2array(T_SSQ(:,7:9)),2);       % 2. Spatial dimension
SSQ(:,3) = mean(table2array(T_SSQ(:,10:13)),2);     % 3. Qualities dimension
SSQ(:,4) = mean(table2array(T_SSQ(:,2:13)),2);      % 4. Global score

% Statistical analysis
p_SSQ = nan(4,1);
[~,p_SSQ(1)] = ttest2(SSQ(1:16,1),SSQ(17:32,1));     % 1. Speech dimension
[~,p_SSQ(2)] = ttest2(SSQ(1:16,2),SSQ(17:32,2));     % 2. Spatial dimension
[~,p_SSQ(3)] = ttest2(SSQ(1:16,3),SSQ(17:32,3));     % 3. Qualities dimension
[~,p_SSQ(4)] = ttest2(SSQ(1:16,4),SSQ(17:32,4));     % 4. Global score

% Table summary
fprintf('SSQ table summary\n')
fprintf('\t\t\t\tUHL\t\t\t\tNH\t\t\t\tp-value\n')
fprintf(sprintf('Speech\t\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n',...
    mean(SSQ(1:16,1)),std(SSQ(1:16,1)),mean(SSQ(17:32,1)),std(SSQ(17:32,1)),p_SSQ(1)))
fprintf(sprintf('Spatial\t\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n',...
    mean(SSQ(1:16,2)),std(SSQ(1:16,2)),mean(SSQ(17:32,2)),std(SSQ(17:32,2)),p_SSQ(2)))
fprintf(sprintf('Qualities\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n',...
    mean(SSQ(1:16,3)),std(SSQ(1:16,3)),mean(SSQ(17:32,3)),std(SSQ(17:32,3)),p_SSQ(3)))
fprintf(sprintf('GLOBAL\t\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n\n\n',...
    mean(SSQ(1:16,4)),std(SSQ(1:16,4)),mean(SSQ(17:32,4)),std(SSQ(17:32,4)),p_SSQ(4)))

% Mean and SEM -- NH in column 1 and UHL in column 2
SSQ_Mean = [mean(SSQ(17:32,1)) , mean(SSQ(1:16,1)) ; ...
            mean(SSQ(17:32,2)) , mean(SSQ(1:16,2)) ; ...
            mean(SSQ(17:32,3)) , mean(SSQ(1:16,3)) ; ...
            mean(SSQ(17:32,4)) , mean(SSQ(1:16,4))];
SSQ_SEM  = [std(SSQ(17:32,1))/sqrt(16) , std(SSQ(1:16,1))/sqrt(16) ; ...
            std(SSQ(17:32,2))/sqrt(16) , std(SSQ(1:16,2))/sqrt(16) ; ...
            std(SSQ(17:32,3))/sqrt(16) , std(SSQ(1:16,3))/sqrt(16) ; ...
            std(SSQ(17:32,4))/sqrt(16) , std(SSQ(1:16,4))/sqrt(16)];

%% SPaRQ subscales: Social behaviour (1-9), Social perceptions (10-19)
% Heffernan et al. (2018). doi: 10.1097/AUD.0000000000000618

% Get average scores
SPaRQ = nan(32,3);
SPaRQ(:,1) = mean(table2array(T_SPaRQ(:,2:10)),2);  % 1. Social behaviour
SPaRQ(:,2) = mean(table2array(T_SPaRQ(:,11:19)),2); % 2. Social perceptions

% Statistical analysis
p_SPaRQ = nan(2,1);
[~,p_SPaRQ(1)] = ttest2(SPaRQ(1:16,1),SPaRQ(17:32,1));     % 1. Behaviour
[~,p_SPaRQ(2)] = ttest2(SPaRQ(1:16,2),SPaRQ(17:32,2));     % 2. Perceptions

% Table summary
fprintf('SPaRQ table summary\n')
fprintf('\t\t\t\tUHL\t\t\t\tNH\t\t\t\tp-value\n')
fprintf(sprintf('Behaviour\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n',...
    mean(SPaRQ(1:16,1)),std(SPaRQ(1:16,1)),mean(SPaRQ(17:32,1)),std(SPaRQ(17:32,1)),p_SPaRQ(1)))
fprintf(sprintf('Perceptions\t\t%1.2f (%1.2f)\t\t%1.2f (%1.2f)\t\t%1.4g\n',...
    mean(SPaRQ(1:16,2)),std(SPaRQ(1:16,2)),mean(SPaRQ(17:32,2)),std(SPaRQ(17:32,2)),p_SPaRQ(2)))

% Mean and SEM -- NH in column 1 and UHL in column 2
SPaRQ_Mean = [mean(SPaRQ(17:32,1)) , mean(SPaRQ(1:16,1)) ; ...
              mean(SPaRQ(17:32,2)) , mean(SPaRQ(1:16,2))];
SPaRQ_SEM  = [std(SPaRQ(17:32,1))/sqrt(16) , std(SPaRQ(1:16,1))/sqrt(16) ; ...
              std(SPaRQ(17:32,2))/sqrt(16) , std(SPaRQ(1:16,2))/sqrt(16)];


%% Figure
% SSQ
FS = 10;
figure('PaperSize',[20 9])
subplot('Position',[0.06 0.08 0.4 0.85]),grid on,hold on,box on
bar(SSQ_Mean)
errorbar((1:4)-0.14,SSQ_Mean(:,1),SSQ_SEM(:,1),'k','LineStyle','none','LineWidth',2)
errorbar((1:4)+0.14,SSQ_Mean(:,2),SSQ_SEM(:,2),'k','LineStyle','none','LineWidth',2)
text(1,8.9,'***','FontSize',FS+10,'HorizontalAlignment','center')
text(2,9.2,'***','FontSize',FS+10,'HorizontalAlignment','center')
text(3,9.2,'***','FontSize',FS+10,'HorizontalAlignment','center')
text(4,9.0,'***','FontSize',FS+10,'HorizontalAlignment','center')
axis([0.5 4.5 0 10.5])
set(gca,'xtick',1:4,'xticklabel',{'Speech','Spatial','Qualities',...
    'Global'},'FontSize',FS)
set(gca,'ytick',0:10,'FontSize',FS)
ylabel('Extreme difficulties                 No difficulties   ',...
    'FontAngle','italic','FontSize',FS)
title('Speech, Spatial and Qualities (SSQ-12)')

% SPaRQ
subplot('Position',[0.59 0.08 0.4 0.85]),grid on,hold on,box on
hv = bar(SPaRQ_Mean);
errorbar((1:2)-0.14,SPaRQ_Mean(:,1),SPaRQ_SEM(:,1),'k','LineStyle','none','LineWidth',2)
errorbar((1:2)+0.14,SPaRQ_Mean(:,2),SPaRQ_SEM(:,2),'k','LineStyle','none','LineWidth',2)
lgnd = legend(hv,'NH','UHL');
set(lgnd,'FontSize',FS-2,'location','NorthEast')
text(1,5.8,'***','FontSize',FS+10,'HorizontalAlignment','center')
text(2,6.1,'***','FontSize',FS+10,'HorizontalAlignment','center')
axis([0.5 2.5 0 10.5])
set(gca,'xtick',1:2,'xticklabel',{sprintf('Social behaviour'),...
    sprintf('Social perception')},'FontSize',FS)
set(gca,'ytick',0:10,'FontSize',FS)
ylabel('No restrictions                Severe restrictions',...
    'FontAngle','italic','FontSize',FS)
title('Social Participation Restrictions (SPaRQ)')

orient tall
print('-dpng','-r600','Figure2')
print('-dpdf','-r600','Figure2')
close all
